"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RecordingManagement = exports.AudioRecordingSamplerateValues = exports.AudioRecordingSamplerate = exports.AudioRecordingCodecType = void 0;
var tslib_1 = require("tslib");
var Characteristic_1 = require("../Characteristic");
var Service_1 = require("../Service");
var debug_1 = (0, tslib_1.__importDefault)(require("debug"));
var tlv = (0, tslib_1.__importStar)(require("../util/tlv"));
var debug = (0, debug_1.default)('HAP-NodeJS:Camera:RecordingManagement');
var AudioBitrate;
(function (AudioBitrate) {
    AudioBitrate[AudioBitrate["VARIABLE"] = 0] = "VARIABLE";
    AudioBitrate[AudioBitrate["CONSTANT"] = 1] = "CONSTANT";
})(AudioBitrate || (AudioBitrate = {}));
var VideoCodecConfigurationTypes;
(function (VideoCodecConfigurationTypes) {
    VideoCodecConfigurationTypes[VideoCodecConfigurationTypes["CODEC_TYPE"] = 1] = "CODEC_TYPE";
    VideoCodecConfigurationTypes[VideoCodecConfigurationTypes["CODEC_PARAMETERS"] = 2] = "CODEC_PARAMETERS";
    VideoCodecConfigurationTypes[VideoCodecConfigurationTypes["ATTRIBUTES"] = 3] = "ATTRIBUTES";
})(VideoCodecConfigurationTypes || (VideoCodecConfigurationTypes = {}));
var VideoCodecParametersTypes;
(function (VideoCodecParametersTypes) {
    VideoCodecParametersTypes[VideoCodecParametersTypes["PROFILE_ID"] = 1] = "PROFILE_ID";
    VideoCodecParametersTypes[VideoCodecParametersTypes["LEVEL"] = 2] = "LEVEL";
    VideoCodecParametersTypes[VideoCodecParametersTypes["BITRATE"] = 3] = "BITRATE";
    VideoCodecParametersTypes[VideoCodecParametersTypes["IFRAME_INTERVAL"] = 4] = "IFRAME_INTERVAL";
})(VideoCodecParametersTypes || (VideoCodecParametersTypes = {}));
var VideoAttributesTypes;
(function (VideoAttributesTypes) {
    VideoAttributesTypes[VideoAttributesTypes["IMAGE_WIDTH"] = 1] = "IMAGE_WIDTH";
    VideoAttributesTypes[VideoAttributesTypes["IMAGE_HEIGHT"] = 2] = "IMAGE_HEIGHT";
    VideoAttributesTypes[VideoAttributesTypes["FRAME_RATE"] = 3] = "FRAME_RATE";
})(VideoAttributesTypes || (VideoAttributesTypes = {}));
var VideoCodecType;
(function (VideoCodecType) {
    VideoCodecType[VideoCodecType["H264"] = 0] = "H264";
})(VideoCodecType || (VideoCodecType = {}));
var SelectedCameraRecordingConfigurationTypes;
(function (SelectedCameraRecordingConfigurationTypes) {
    SelectedCameraRecordingConfigurationTypes[SelectedCameraRecordingConfigurationTypes["SELECTED_GENERAL_CONFIGURATION"] = 1] = "SELECTED_GENERAL_CONFIGURATION";
    SelectedCameraRecordingConfigurationTypes[SelectedCameraRecordingConfigurationTypes["SELECTED_VIDEO_CONFIGURATION"] = 2] = "SELECTED_VIDEO_CONFIGURATION";
    SelectedCameraRecordingConfigurationTypes[SelectedCameraRecordingConfigurationTypes["SELECTED_AUDIO_CONFIGURATION"] = 3] = "SELECTED_AUDIO_CONFIGURATION";
})(SelectedCameraRecordingConfigurationTypes || (SelectedCameraRecordingConfigurationTypes = {}));
var AudioRecordingCodecType;
(function (AudioRecordingCodecType) {
    AudioRecordingCodecType[AudioRecordingCodecType["AAC_LC"] = 0] = "AAC_LC";
    AudioRecordingCodecType[AudioRecordingCodecType["AAC_ELD"] = 1] = "AAC_ELD";
})(AudioRecordingCodecType = exports.AudioRecordingCodecType || (exports.AudioRecordingCodecType = {}));
var AudioRecordingSamplerate;
(function (AudioRecordingSamplerate) {
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_8"] = 0] = "KHZ_8";
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_16"] = 1] = "KHZ_16";
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_24"] = 2] = "KHZ_24";
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_32"] = 3] = "KHZ_32";
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_44_1"] = 4] = "KHZ_44_1";
    AudioRecordingSamplerate[AudioRecordingSamplerate["KHZ_48"] = 5] = "KHZ_48";
})(AudioRecordingSamplerate = exports.AudioRecordingSamplerate || (exports.AudioRecordingSamplerate = {}));
exports.AudioRecordingSamplerateValues = {
    0: 8,
    1: 16,
    2: 24,
    3: 32,
    4: 44.1,
    5: 48,
};
var SupportedVideoRecordingConfigurationTypes;
(function (SupportedVideoRecordingConfigurationTypes) {
    SupportedVideoRecordingConfigurationTypes[SupportedVideoRecordingConfigurationTypes["VIDEO_CODEC_CONFIGURATION"] = 1] = "VIDEO_CODEC_CONFIGURATION";
})(SupportedVideoRecordingConfigurationTypes || (SupportedVideoRecordingConfigurationTypes = {}));
var SupportedCameraRecordingConfigurationTypes;
(function (SupportedCameraRecordingConfigurationTypes) {
    SupportedCameraRecordingConfigurationTypes[SupportedCameraRecordingConfigurationTypes["PREBUFFER_LENGTH"] = 1] = "PREBUFFER_LENGTH";
    SupportedCameraRecordingConfigurationTypes[SupportedCameraRecordingConfigurationTypes["EVENT_TRIGGER_OPTIONS"] = 2] = "EVENT_TRIGGER_OPTIONS";
    SupportedCameraRecordingConfigurationTypes[SupportedCameraRecordingConfigurationTypes["MEDIA_CONTAINER_CONFIGURATIONS"] = 3] = "MEDIA_CONTAINER_CONFIGURATIONS";
})(SupportedCameraRecordingConfigurationTypes || (SupportedCameraRecordingConfigurationTypes = {}));
var MediaContainerConfigurationTypes;
(function (MediaContainerConfigurationTypes) {
    MediaContainerConfigurationTypes[MediaContainerConfigurationTypes["MEDIA_CONTAINER_TYPE"] = 1] = "MEDIA_CONTAINER_TYPE";
    MediaContainerConfigurationTypes[MediaContainerConfigurationTypes["MEDIA_CONTAINER_PARAMETERS"] = 2] = "MEDIA_CONTAINER_PARAMETERS";
})(MediaContainerConfigurationTypes || (MediaContainerConfigurationTypes = {}));
var MediaContainerParameterTypes;
(function (MediaContainerParameterTypes) {
    MediaContainerParameterTypes[MediaContainerParameterTypes["FRAGMENT_LENGTH"] = 1] = "FRAGMENT_LENGTH";
})(MediaContainerParameterTypes || (MediaContainerParameterTypes = {}));
var AudioCodecParametersTypes;
(function (AudioCodecParametersTypes) {
    AudioCodecParametersTypes[AudioCodecParametersTypes["CHANNEL"] = 1] = "CHANNEL";
    AudioCodecParametersTypes[AudioCodecParametersTypes["BIT_RATE"] = 2] = "BIT_RATE";
    AudioCodecParametersTypes[AudioCodecParametersTypes["SAMPLE_RATE"] = 3] = "SAMPLE_RATE";
    AudioCodecParametersTypes[AudioCodecParametersTypes["MAX_AUDIO_BITRATE"] = 4] = "MAX_AUDIO_BITRATE"; // only present in selected audio codec parameters tlv
})(AudioCodecParametersTypes || (AudioCodecParametersTypes = {}));
var AudioCodecConfigurationTypes;
(function (AudioCodecConfigurationTypes) {
    AudioCodecConfigurationTypes[AudioCodecConfigurationTypes["CODEC_TYPE"] = 1] = "CODEC_TYPE";
    AudioCodecConfigurationTypes[AudioCodecConfigurationTypes["CODEC_PARAMETERS"] = 2] = "CODEC_PARAMETERS";
})(AudioCodecConfigurationTypes || (AudioCodecConfigurationTypes = {}));
var SupportedAudioRecordingConfigurationTypes;
(function (SupportedAudioRecordingConfigurationTypes) {
    SupportedAudioRecordingConfigurationTypes[SupportedAudioRecordingConfigurationTypes["AUDIO_CODEC_CONFIGURATION"] = 1] = "AUDIO_CODEC_CONFIGURATION";
})(SupportedAudioRecordingConfigurationTypes || (SupportedAudioRecordingConfigurationTypes = {}));
var RecordingManagement = /** @class */ (function () {
    function RecordingManagement(options, delegate, service) {
        this.selectedConfiguration = ""; // base64 representation of the currently selected configuration
        this.recordingAudioActive = true;
        this.active = false;
        this.delegate = delegate;
        this.service = service || this.constructService();
        this.setupServiceHandlers();
        this.supportedCameraRecordingConfiguration = RecordingManagement._supportedCameraRecordingConfiguration(options);
        this.supportedVideoRecordingConfiguration = RecordingManagement._supportedVideoRecordingConfiguration(options.video);
        this.supportedAudioRecordingConfiguration = this._supportedAudioStreamConfiguration(options.audio);
    }
    RecordingManagement._supportedCameraRecordingConfiguration = function (options) {
        var eventTriggerOptions = Buffer.alloc(8);
        eventTriggerOptions.writeInt32LE(options.eventTriggerOptions, 0);
        var prebufferLength = Buffer.alloc(4);
        prebufferLength.writeInt32LE(options.prebufferLength, 0);
        return tlv.encode(1 /* PREBUFFER_LENGTH */, prebufferLength, 2 /* EVENT_TRIGGER_OPTIONS */, eventTriggerOptions, 3 /* MEDIA_CONTAINER_CONFIGURATIONS */, options.mediaContainerConfigurations.map(function (config) {
            var fragmentLength = Buffer.alloc(4);
            fragmentLength.writeInt32LE(config.fragmentLength, 0);
            return tlv.encode(1 /* MEDIA_CONTAINER_TYPE */, config.type, 2 /* MEDIA_CONTAINER_PARAMETERS */, tlv.encode(1 /* FRAGMENT_LENGTH */, fragmentLength));
        })).toString('base64');
    };
    RecordingManagement._supportedVideoRecordingConfiguration = function (videoOptions) {
        if (!videoOptions.codec) {
            throw new Error('Video codec cannot be undefined');
        }
        if (!videoOptions.resolutions) {
            throw new Error('Video resolutions cannot be undefined');
        }
        var codecParameters = tlv.encode(1 /* PROFILE_ID */, videoOptions.codec.profiles, 2 /* LEVEL */, videoOptions.codec.levels);
        var videoStreamConfiguration = tlv.encode(1 /* CODEC_TYPE */, 0 /* H264 */, 2 /* CODEC_PARAMETERS */, codecParameters, 3 /* ATTRIBUTES */, videoOptions.resolutions.map(function (resolution) {
            if (resolution.length != 3) {
                throw new Error('Unexpected video resolution');
            }
            var width = Buffer.alloc(2);
            var height = Buffer.alloc(2);
            var frameRate = Buffer.alloc(1);
            width.writeUInt16LE(resolution[0], 0);
            height.writeUInt16LE(resolution[1], 0);
            frameRate.writeUInt8(resolution[2], 0);
            return tlv.encode(1 /* IMAGE_WIDTH */, width, 2 /* IMAGE_HEIGHT */, height, 3 /* FRAME_RATE */, frameRate);
        }));
        return tlv.encode(1 /* VIDEO_CODEC_CONFIGURATION */, videoStreamConfiguration).toString('base64');
    };
    RecordingManagement.prototype._supportedAudioStreamConfiguration = function (audioOptions) {
        // Only AAC-ELD and OPUS are accepted by iOS currently, and we need to give it something it will accept
        // for it to start the video stream.
        var supportedCodecs = (audioOptions && audioOptions.codecs) || [];
        if (supportedCodecs.length === 0) { // Fake a Codec if we haven't got anything
            debug("Client doesn't support any audio codec that HomeKit supports.");
            supportedCodecs.push({
                type: 0 /* AAC_LC */,
                bitrateMode: 0,
                audioChannels: 1,
                samplerate: [1 /* KHZ_16 */, 2 /* KHZ_24 */], // 16 and 24 must be supported
            });
        }
        var codecConfigurations = supportedCodecs.map(function (codec) {
            var providedSamplerates = (typeof codec.samplerate === "number" ? [codec.samplerate] : codec.samplerate);
            if (providedSamplerates.length === 0) {
                throw new Error("Audio samplerate cannot be empty!");
            }
            var audioParameters = tlv.encode(1 /* CHANNEL */, Math.max(1, codec.audioChannels || 1), 2 /* BIT_RATE */, codec.bitrateMode || 0 /* VARIABLE */, 3 /* SAMPLE_RATE */, providedSamplerates);
            return tlv.encode(1 /* CODEC_TYPE */, codec.type, 2 /* CODEC_PARAMETERS */, audioParameters);
        });
        return tlv.encode(1 /* AUDIO_CODEC_CONFIGURATION */, codecConfigurations).toString("base64");
    };
    RecordingManagement.prototype.constructService = function () {
        var _this = this;
        var managementService = new Service_1.Service.CameraRecordingManagement('', '');
        managementService.getCharacteristic(Characteristic_1.Characteristic.Active)
            .on('get', function (callback) {
            callback(null, _this.active);
        })
            .on('set', function (value, callback) {
            _this.active = !!value;
            callback();
        });
        managementService.getCharacteristic(Characteristic_1.Characteristic.RecordingAudioActive)
            .on('get', function (callback) {
            callback(null, _this.recordingAudioActive);
        })
            .on('set', function (value, callback) {
            _this.recordingAudioActive = !!value;
            callback();
        });
        managementService.getCharacteristic(Characteristic_1.Characteristic.SupportedCameraRecordingConfiguration)
            .on('get', function (callback) {
            callback(null, _this.supportedCameraRecordingConfiguration);
        });
        managementService.getCharacteristic(Characteristic_1.Characteristic.SupportedVideoRecordingConfiguration)
            .on('get', function (callback) {
            callback(null, _this.supportedVideoRecordingConfiguration);
        });
        managementService.getCharacteristic(Characteristic_1.Characteristic.SupportedAudioRecordingConfiguration)
            .on('get', function (callback) {
            callback(null, _this.supportedAudioRecordingConfiguration);
        });
        return managementService;
    };
    RecordingManagement.prototype.setupServiceHandlers = function () {
        var _this = this;
        this.service.getCharacteristic(Characteristic_1.Characteristic.SelectedCameraRecordingConfiguration)
            .on("get" /* GET */, function (callback) {
            callback(null, _this.selectedConfiguration);
        })
            .on("set" /* SET */, function (value, callback) {
            var _a, _b;
            _this.selectedConfiguration = value.toString();
            callback();
            (_b = (_a = _this.delegate).prepareRecording) === null || _b === void 0 ? void 0 : _b.call(_a, _this.getSelectedConfiguration());
        });
    };
    RecordingManagement.prototype.getSelectedConfiguration = function () {
        var decoded = tlv.decode(Buffer.from(this.selectedConfiguration, 'base64'));
        var recording = tlv.decode(decoded[1 /* SELECTED_GENERAL_CONFIGURATION */]);
        var video = tlv.decode(decoded[2 /* SELECTED_VIDEO_CONFIGURATION */]);
        var audio = tlv.decode(decoded[3 /* SELECTED_AUDIO_CONFIGURATION */]);
        var vcodec = video[1 /* CODEC_TYPE */][0];
        var vparameters = tlv.decode(video[2 /* CODEC_PARAMETERS */]);
        var vattributes = tlv.decode(video[3 /* ATTRIBUTES */]);
        var width = vattributes[1 /* IMAGE_WIDTH */].readInt16LE(0);
        var height = vattributes[2 /* IMAGE_HEIGHT */].readInt16LE(0);
        var framerate = vattributes[3 /* FRAME_RATE */][0];
        var profile = vparameters[1 /* PROFILE_ID */][0];
        var level = vparameters[2 /* LEVEL */][0];
        var videoBitrate = vparameters[3 /* BITRATE */].readInt32LE(0);
        var iFrameInterval = vparameters[4 /* IFRAME_INTERVAL */].readInt32LE(0);
        var prebufferLength = recording[1 /* PREBUFFER_LENGTH */].readInt32LE(0);
        var mediaContainerConfiguration = tlv.decode(recording[3 /* MEDIA_CONTAINER_CONFIGURATIONS */]);
        var containerType = mediaContainerConfiguration[1 /* MEDIA_CONTAINER_TYPE */][0];
        var mediaContainerParameters = tlv.decode(mediaContainerConfiguration[2 /* MEDIA_CONTAINER_PARAMETERS */]);
        var fragmentLength = mediaContainerParameters[1 /* FRAGMENT_LENGTH */].readInt32LE(0);
        var acodec = audio[1 /* CODEC_TYPE */][0];
        var audioParameters = tlv.decode(audio[2 /* CODEC_PARAMETERS */]);
        var audioChannels = audioParameters[1 /* CHANNEL */][0];
        var samplerate = audioParameters[3 /* SAMPLE_RATE */][0];
        var audioBitrateMode = audioParameters[2 /* BIT_RATE */][0];
        var audioBitrate = audioParameters[4 /* MAX_AUDIO_BITRATE */].readUInt32LE(0);
        return {
            mediaContainerConfiguration: {
                prebufferLength: prebufferLength,
                type: containerType,
                fragmentLength: fragmentLength,
            },
            videoCodec: {
                type: vcodec,
                bitrate: videoBitrate,
                level: level,
                profile: profile,
                resolution: [width, height, framerate],
                iFrameInterval: iFrameInterval,
            },
            audioCodec: {
                audioChannels: audioChannels,
                type: acodec,
                samplerate: samplerate,
                bitrateMode: audioBitrateMode,
                bitrate: audioBitrate,
            },
        };
    };
    RecordingManagement.prototype.getService = function () {
        return this.service;
    };
    return RecordingManagement;
}());
exports.RecordingManagement = RecordingManagement;
//# sourceMappingURL=RecordingManagement.js.map