"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CameraController = exports.CameraControllerEvents = void 0;
var tslib_1 = require("tslib");
var crypto_1 = (0, tslib_1.__importDefault)(require("crypto"));
var debug_1 = (0, tslib_1.__importDefault)(require("debug"));
var events_1 = require("events");
var camera_1 = require("../camera");
var Characteristic_1 = require("../Characteristic");
var datastream_1 = require("../datastream");
var definitions_1 = require("../definitions");
var Service_1 = require("../Service");
var debug = (0, debug_1.default)("HAP-NodeJS:Camera:Controller");
var CameraControllerEvents;
(function (CameraControllerEvents) {
    /**
     *  Emitted when the mute state or the volume changed. The Apple Home App typically does not set those values
     *  except the mute state. When you adjust the volume in the Camera view it will reset the muted state if it was set previously.
     *  The value of volume has nothing to do with the volume slider in the Camera view of the Home app.
     */
    CameraControllerEvents["MICROPHONE_PROPERTIES_CHANGED"] = "microphone-change";
    /**
     * Emitted when the mute state or the volume changed. The Apple Home App typically does not set those values
     * except the mute state. When you unmute the device microphone it will reset the mute state if it was set previously.
     */
    CameraControllerEvents["SPEAKER_PROPERTIES_CHANGED"] = "speaker-change";
})(CameraControllerEvents = exports.CameraControllerEvents || (exports.CameraControllerEvents = {}));
/**
 * Everything needed to expose a HomeKit Camera.
 */
var CameraController = /** @class */ (function (_super) {
    (0, tslib_1.__extends)(CameraController, _super);
    function CameraController(options, legacyMode) {
        if (legacyMode === void 0) { legacyMode = false; }
        var _a, _b;
        var _this = _super.call(this) || this;
        _this.legacyMode = false;
        /**
         * @private
         */
        _this.streamManagements = [];
        _this.microphoneMuted = false;
        _this.microphoneVolume = 100;
        _this.speakerMuted = false;
        _this.speakerVolume = 100;
        _this.connectionMap = new Map();
        _this.homekitCameraActive = false;
        _this.eventSnapshotsActive = false;
        _this.periodicSnapshotsActive = false;
        _this.streamCount = Math.max(1, options.cameraStreamCount || 1);
        _this.delegate = options.delegate;
        _this.streamingOptions = options.streamingOptions;
        _this.recordingOptions = (_a = options.recording) === null || _a === void 0 ? void 0 : _a.options;
        _this.recordingDelegate = (_b = options.recording) === null || _b === void 0 ? void 0 : _b.delegate;
        _this.legacyMode = legacyMode; // legacy mode will prent from Microphone and Speaker services to get created to avoid collisions
        return _this;
    }
    /**
     * @private
     */
    CameraController.prototype.controllerId = function () {
        return "camera" /* CAMERA */;
    };
    // ----------------------------------- STREAM API ------------------------------------
    /**
     * Call this method if you want to forcefully suspend an ongoing streaming session.
     * This would be adequate if the the rtp server or media encoding encountered an unexpected error.
     *
     * @param sessionId {SessionIdentifier} - id of the current ongoing streaming session
     */
    CameraController.prototype.forceStopStreamingSession = function (sessionId) {
        this.streamManagements.forEach(function (management) {
            if (management.sessionIdentifier === sessionId) {
                management.forceStop();
            }
        });
    };
    CameraController.generateSynchronisationSource = function () {
        var ssrc = crypto_1.default.randomBytes(4); // range [-2.14748e+09 - 2.14748e+09]
        ssrc[0] = 0;
        return ssrc.readInt32BE(0);
    };
    // ----------------------------- MICROPHONE/SPEAKER API ------------------------------
    CameraController.prototype.setMicrophoneMuted = function (muted) {
        if (muted === void 0) { muted = true; }
        if (!this.microphoneService) {
            return;
        }
        this.microphoneMuted = muted;
        this.microphoneService.updateCharacteristic(Characteristic_1.Characteristic.Mute, muted);
    };
    CameraController.prototype.setMicrophoneVolume = function (volume) {
        if (!this.microphoneService) {
            return;
        }
        this.microphoneVolume = volume;
        this.microphoneService.updateCharacteristic(Characteristic_1.Characteristic.Volume, volume);
    };
    CameraController.prototype.setSpeakerMuted = function (muted) {
        if (muted === void 0) { muted = true; }
        if (!this.speakerService) {
            return;
        }
        this.speakerMuted = muted;
        this.speakerService.updateCharacteristic(Characteristic_1.Characteristic.Mute, muted);
    };
    CameraController.prototype.setSpeakerVolume = function (volume) {
        if (!this.speakerService) {
            return;
        }
        this.speakerVolume = volume;
        this.speakerService.updateCharacteristic(Characteristic_1.Characteristic.Volume, volume);
    };
    CameraController.prototype.emitMicrophoneChange = function () {
        this.emit("microphone-change" /* MICROPHONE_PROPERTIES_CHANGED */, this.microphoneMuted, this.microphoneVolume);
    };
    CameraController.prototype.emitSpeakerChange = function () {
        this.emit("speaker-change" /* SPEAKER_PROPERTIES_CHANGED */, this.speakerMuted, this.speakerVolume);
    };
    // -----------------------------------------------------------------------------------
    /**
     * @private
     */
    CameraController.prototype.constructServices = function () {
        var _a, _b;
        for (var i = 0; i < this.streamCount; i++) {
            var rtp = new camera_1.RTPStreamManagement(i, this.streamingOptions, this.delegate);
            this.streamManagements.push(rtp);
            if (this.recordingOptions) {
                rtp.getService().setCharacteristic(Characteristic_1.Characteristic.Active, 1);
            }
        }
        if (!this.legacyMode && this.streamingOptions.audio) {
            // In theory the Microphone Service is a necessity. In practice its not. lol. So we just add it if the user wants to support audio
            this.microphoneService = new Service_1.Service.Microphone('', '');
            this.microphoneService.setCharacteristic(Characteristic_1.Characteristic.Volume, this.microphoneVolume);
            if (this.streamingOptions.audio.twoWayAudio) {
                this.speakerService = new Service_1.Service.Speaker('', '');
                this.speakerService.setCharacteristic(Characteristic_1.Characteristic.Volume, this.speakerVolume);
            }
        }
        if (this.recordingOptions) {
            this.cameraOperatingModeService = new Service_1.Service.CameraOperatingMode('', '');
            this.recordingManagement = new camera_1.RecordingManagement(this.recordingOptions, this.recordingDelegate);
            this.dataStreamManagement = new datastream_1.DataStreamManagement();
            if (this.recordingOptions.motionService) {
                this.motionService = new definitions_1.MotionSensor('', '');
                this.motionService.setCharacteristic(Characteristic_1.Characteristic.Active, 1);
                this.recordingManagement.getService().addLinkedService(this.motionService);
            }
            this.recordingManagement.getService().addLinkedService(this.dataStreamManagement.getService());
        }
        var serviceMap = {
            microphone: this.microphoneService,
            speaker: this.speakerService,
            cameraOperatingMode: this.cameraOperatingModeService,
            cameraEventRecordingManagement: (_a = this.recordingManagement) === null || _a === void 0 ? void 0 : _a.getService(),
            dataStreamTransportManagement: (_b = this.dataStreamManagement) === null || _b === void 0 ? void 0 : _b.getService(),
            motionService: this.motionService,
        };
        this.streamManagements.forEach(function (management, index) { return serviceMap[CameraController.STREAM_MANAGEMENT + index] = management.getService(); });
        return serviceMap;
    };
    /**
     * @private
     */
    CameraController.prototype.initWithServices = function (serviceMap) {
        var _a;
        var modifiedServiceMap = false;
        for (var i = 0; true; i++) {
            var streamManagementService = serviceMap[CameraController.STREAM_MANAGEMENT + i];
            if (i < this.streamCount) {
                if (streamManagementService) { // normal init
                    this.streamManagements.push(new camera_1.RTPStreamManagement(i, this.streamingOptions, this.delegate, streamManagementService));
                }
                else { // stream count got bigger, we need to create a new service
                    var management = new camera_1.RTPStreamManagement(i, this.streamingOptions, this.delegate);
                    this.streamManagements.push(management);
                    serviceMap[CameraController.STREAM_MANAGEMENT + i] = management.getService();
                    modifiedServiceMap = true;
                }
            }
            else {
                if (streamManagementService) { // stream count got reduced, we need to remove old service
                    delete serviceMap[CameraController.STREAM_MANAGEMENT + i];
                    modifiedServiceMap = true;
                }
                else {
                    break; // we finished counting and we got no saved service; we are finished
                }
            }
        }
        // MICROPHONE
        if (!this.legacyMode && this.streamingOptions.audio) { // microphone should be present
            if (serviceMap.microphone) {
                this.microphoneService = serviceMap.microphone;
            }
            else {
                // microphone wasn't created yet => create a new one
                this.microphoneService = new Service_1.Service.Microphone('', '');
                this.microphoneService.setCharacteristic(Characteristic_1.Characteristic.Volume, this.microphoneVolume);
                serviceMap.microphone = this.microphoneService;
                modifiedServiceMap = true;
            }
        }
        else if (serviceMap.microphone) { // microphone service supplied, though settings seemed to have changed
            // we need to remove it
            delete serviceMap.microphone;
            modifiedServiceMap = true;
        }
        // SPEAKER
        if (!this.legacyMode && ((_a = this.streamingOptions.audio) === null || _a === void 0 ? void 0 : _a.twoWayAudio)) { // speaker should be present
            if (serviceMap.speaker) {
                this.speakerService = serviceMap.speaker;
            }
            else {
                // speaker wasn't created yet => create a new one
                this.speakerService = new Service_1.Service.Speaker('', '');
                this.speakerService.setCharacteristic(Characteristic_1.Characteristic.Volume, this.speakerVolume);
                serviceMap.speaker = this.speakerService;
                modifiedServiceMap = true;
            }
        }
        else if (serviceMap.speaker) { // speaker service supplied, though settings seemed to have changed
            // we need to remove it
            delete serviceMap.speaker;
            modifiedServiceMap = true;
        }
        if (this.recordingOptions) {
            if (serviceMap.cameraOperatingMode) {
                this.cameraOperatingModeService = serviceMap.cameraOperatingMode;
            }
            else {
                this.cameraOperatingModeService = new Service_1.Service.CameraOperatingMode('', '');
                serviceMap.cameraOperatingMode = this.cameraOperatingModeService;
                modifiedServiceMap = true;
            }
            if (serviceMap.cameraEventRecordingManagement) {
                this.recordingManagement = new camera_1.RecordingManagement(this.recordingOptions, this.recordingDelegate, serviceMap.cameraEventRecordingManagement);
            }
            else {
                this.recordingManagement = new camera_1.RecordingManagement(this.recordingOptions, this.recordingDelegate);
                serviceMap.cameraEventRecordingManagement = this.recordingManagement.getService();
                modifiedServiceMap = true;
            }
            if (serviceMap.dataStreamTransportManagement) {
                this.dataStreamManagement = new datastream_1.DataStreamManagement(serviceMap.dataStreamTransportManagement);
            }
            else {
                this.dataStreamManagement = new datastream_1.DataStreamManagement();
                serviceMap.dataStreamTransportManagement = this.dataStreamManagement.getService();
                modifiedServiceMap = true;
            }
            if (!this.recordingOptions.motionService) {
                if (serviceMap.motionService) {
                    delete serviceMap.motionService;
                    modifiedServiceMap = true;
                }
            }
            else {
                if (!serviceMap.motionService) {
                    this.motionService = new definitions_1.MotionSensor('', '');
                    serviceMap.motionService = this.motionService;
                    modifiedServiceMap = true;
                }
            }
        }
        else {
            if (serviceMap.cameraOperatingMode) {
                delete serviceMap.cameraOperatingMode;
                modifiedServiceMap = true;
            }
            if (serviceMap.cameraEventRecordingManagement) {
                delete serviceMap.cameraEventRecordingManagement;
                modifiedServiceMap = true;
            }
            if (serviceMap.dataStreamTransportManagement) {
                delete serviceMap.dataStreamTransportManagement;
                modifiedServiceMap = true;
            }
            if (serviceMap.motionService) {
                delete serviceMap.motionService;
                modifiedServiceMap = true;
            }
        }
        if (this.migrateFromDoorbell(serviceMap)) {
            modifiedServiceMap = true;
        }
        if (modifiedServiceMap) { // serviceMap must only be returned if anything actually changed
            return serviceMap;
        }
    };
    // overwritten in DoorbellController (to avoid cyclic dependencies, i hate typescript for that)
    CameraController.prototype.migrateFromDoorbell = function (serviceMap) {
        if (serviceMap.doorbell) { // See NOTICE in DoorbellController
            delete serviceMap.doorbell;
            return true;
        }
        return false;
    };
    /**
     * @private
     */
    CameraController.prototype.configureServices = function () {
        var _this = this;
        if (this.microphoneService) {
            this.microphoneService.getCharacteristic(Characteristic_1.Characteristic.Mute)
                .on("get" /* GET */, function (callback) {
                callback(undefined, _this.microphoneMuted);
            })
                .on("set" /* SET */, function (value, callback) {
                _this.microphoneMuted = value;
                callback();
                _this.emitMicrophoneChange();
            });
            this.microphoneService.getCharacteristic(Characteristic_1.Characteristic.Volume)
                .on("get" /* GET */, function (callback) {
                callback(undefined, _this.microphoneVolume);
            })
                .on("set" /* SET */, function (value, callback) {
                _this.microphoneVolume = value;
                callback();
                _this.emitMicrophoneChange();
            });
        }
        if (this.speakerService) {
            this.speakerService.getCharacteristic(Characteristic_1.Characteristic.Mute)
                .on("get" /* GET */, function (callback) {
                callback(undefined, _this.speakerMuted);
            })
                .on("set" /* SET */, function (value, callback) {
                _this.speakerMuted = value;
                callback();
                _this.emitSpeakerChange();
            });
            this.speakerService.getCharacteristic(Characteristic_1.Characteristic.Volume)
                .on("get" /* GET */, function (callback) {
                callback(undefined, _this.speakerVolume);
            })
                .on("set" /* SET */, function (value, callback) {
                _this.speakerVolume = value;
                callback();
                _this.emitSpeakerChange();
            });
        }
        if (this.cameraOperatingModeService) {
            this.cameraOperatingModeService.getCharacteristic(Characteristic_1.Characteristic.EventSnapshotsActive)
                .on('get', function (callback) {
                callback(null, _this.eventSnapshotsActive);
            })
                .on('set', function (value, callback) {
                _this.eventSnapshotsActive = !!value;
                callback();
            });
            this.cameraOperatingModeService.getCharacteristic(Characteristic_1.Characteristic.HomeKitCameraActive)
                .on('get', function (callback) {
                callback(null, _this.homekitCameraActive);
            })
                .on('set', function (value, callback) {
                _this.homekitCameraActive = !!value;
                callback();
            });
            this.cameraOperatingModeService.getCharacteristic(Characteristic_1.Characteristic.PeriodicSnapshotsActive)
                .on('get', function (callback) {
                callback(null, _this.periodicSnapshotsActive);
            })
                .on('set', function (value, callback) {
                _this.periodicSnapshotsActive = !!value;
                callback();
            });
        }
        if (this.dataStreamManagement) {
            this.dataStreamManagement
                .onRequestMessage("dataSend" /* DATA_SEND */, "open" /* OPEN */, this.handleDataSendOpen.bind(this))
                .onEventMessage("dataSend" /* DATA_SEND */, "close" /* CLOSE */, this.handleDataSendClose.bind(this))
                .onServerEvent("connection-closed" /* CONNECTION_CLOSED */, this.handleDataStreamConnectionClosed.bind(this));
        }
    };
    CameraController.prototype.handleDataSendOpen = function (connection, id, message) {
        var e_1, _a;
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var streamId, generator, first, maxChunk, dataSequenceNumber, generator_1, generator_1_1, fragment, wasFirst, offset, dataChunkSequenceNumber, data, isLastDataChunk, event, e_1_1, e_2;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        streamId = message.streamId;
                        generator = this.recordingDelegate.handleFragmentsRequests(this.recordingManagement.getSelectedConfiguration());
                        this.connectionMap.set(streamId, { generator: generator, connection: connection });
                        first = true;
                        maxChunk = 0x40000;
                        _b.label = 1;
                    case 1:
                        _b.trys.push([1, 14, 15, 16]);
                        dataSequenceNumber = 1;
                        _b.label = 2;
                    case 2:
                        _b.trys.push([2, 7, 8, 13]);
                        generator_1 = (0, tslib_1.__asyncValues)(generator);
                        _b.label = 3;
                    case 3: return [4 /*yield*/, generator_1.next()];
                    case 4:
                        if (!(generator_1_1 = _b.sent(), !generator_1_1.done)) return [3 /*break*/, 6];
                        fragment = generator_1_1.value;
                        wasFirst = first;
                        if (first) {
                            first = false;
                            connection.sendResponse("dataSend" /* DATA_SEND */, "open" /* OPEN */, id, datastream_1.HDSStatus.SUCCESS, {
                                status: datastream_1.HDSStatus.SUCCESS,
                            });
                        }
                        offset = 0;
                        dataChunkSequenceNumber = 1;
                        while (offset < fragment.length) {
                            data = fragment.slice(offset, offset + maxChunk);
                            offset += data.length;
                            isLastDataChunk = offset >= fragment.length;
                            event = {
                                streamId: streamId,
                                packets: [
                                    {
                                        metadata: {
                                            dataType: wasFirst ? 'mediaInitialization' : 'mediaFragment',
                                            dataSequenceNumber: dataSequenceNumber,
                                            isLastDataChunk: isLastDataChunk,
                                            dataChunkSequenceNumber: dataChunkSequenceNumber,
                                        },
                                        data: data,
                                    }
                                ]
                            };
                            connection.sendEvent("dataSend" /* DATA_SEND */, "data" /* DATA */, event);
                            dataChunkSequenceNumber++;
                        }
                        dataSequenceNumber++;
                        _b.label = 5;
                    case 5: return [3 /*break*/, 3];
                    case 6: return [3 /*break*/, 13];
                    case 7:
                        e_1_1 = _b.sent();
                        e_1 = { error: e_1_1 };
                        return [3 /*break*/, 13];
                    case 8:
                        _b.trys.push([8, , 11, 12]);
                        if (!(generator_1_1 && !generator_1_1.done && (_a = generator_1.return))) return [3 /*break*/, 10];
                        return [4 /*yield*/, _a.call(generator_1)];
                    case 9:
                        _b.sent();
                        _b.label = 10;
                    case 10: return [3 /*break*/, 12];
                    case 11:
                        if (e_1) throw e_1.error;
                        return [7 /*endfinally*/];
                    case 12: return [7 /*endfinally*/];
                    case 13: return [3 /*break*/, 16];
                    case 14:
                        e_2 = _b.sent();
                        return [3 /*break*/, 16];
                    case 15:
                        if (first) {
                            connection.sendResponse("dataSend" /* DATA_SEND */, "open" /* OPEN */, id, datastream_1.HDSStatus.PROTOCOL_SPECIFIC_ERROR, {
                                status: datastream_1.HDSStatus.PROTOCOL_SPECIFIC_ERROR,
                            });
                        }
                        return [7 /*endfinally*/];
                    case 16: return [2 /*return*/];
                }
            });
        });
    };
    CameraController.prototype.handleDataSendClose = function (connection, message) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var streamId, entry, generator;
            return (0, tslib_1.__generator)(this, function (_a) {
                streamId = message.streamId;
                entry = this.connectionMap.get(streamId);
                if (!entry)
                    return [2 /*return*/];
                this.connectionMap.delete(streamId);
                generator = entry.generator;
                generator.throw('dataSend close');
                return [2 /*return*/];
            });
        });
    };
    CameraController.prototype.handleDataStreamConnectionClosed = function (closedConnection) {
        var e_3, _a;
        try {
            for (var _b = (0, tslib_1.__values)(this.connectionMap.entries()), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = (0, tslib_1.__read)(_c.value, 2), key = _d[0], _e = _d[1], generator = _e.generator, connection = _e.connection;
                if (connection === closedConnection) {
                    this.connectionMap.delete(key);
                    generator.throw('connection closed');
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_3) throw e_3.error; }
        }
    };
    /**
     * @private
     */
    CameraController.prototype.handleControllerRemoved = function () {
        var e_4, _a;
        this.handleFactoryReset();
        try {
            for (var _b = (0, tslib_1.__values)(this.streamManagements), _c = _b.next(); !_c.done; _c = _b.next()) {
                var management = _c.value;
                management.destroy();
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_4) throw e_4.error; }
        }
        this.streamManagements.splice(0, this.streamManagements.length);
        this.microphoneService = undefined;
        this.speakerService = undefined;
        this.removeAllListeners();
    };
    /**
     * @private
     */
    CameraController.prototype.handleFactoryReset = function () {
        this.streamManagements.forEach(function (management) { return management.handleFactoryReset(); });
        this.microphoneMuted = false;
        this.microphoneVolume = 100;
        this.speakerMuted = false;
        this.speakerVolume = 100;
    };
    /**
     * @private
     */
    CameraController.prototype.handleSnapshotRequest = function (height, width, accessoryName, reason) {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var timeout = setTimeout(function () {
                console.warn("[" + accessoryName + "] The image snapshot handler for the given accessory is slow to respond! See https://git.io/JtMGR for more info.");
                timeout = setTimeout(function () {
                    timeout = undefined;
                    console.warn("[" + accessoryName + "] The image snapshot handler for the given accessory didn't respond at all! See https://git.io/JtMGR for more info.");
                    reject(-70408 /* OPERATION_TIMED_OUT */);
                }, 17000);
                timeout.unref();
            }, 5000);
            timeout.unref();
            try {
                _this.delegate.handleSnapshotRequest({
                    height: height,
                    width: width,
                    reason: reason,
                }, function (error, buffer) {
                    if (!timeout) {
                        return;
                    }
                    else {
                        clearTimeout(timeout);
                        timeout = undefined;
                    }
                    if (error) {
                        if (typeof error === "number") {
                            reject(error);
                        }
                        else {
                            debug("[%s] Error getting snapshot: %s", accessoryName, error.stack);
                            reject(-70402 /* SERVICE_COMMUNICATION_FAILURE */);
                        }
                        return;
                    }
                    if (!buffer || buffer.length === 0) {
                        console.warn("[" + accessoryName + "] Snapshot request handler provided empty image buffer!");
                        reject(-70402 /* SERVICE_COMMUNICATION_FAILURE */);
                    }
                    else {
                        resolve(buffer);
                    }
                });
            }
            catch (error) {
                if (!timeout) {
                    return;
                }
                else {
                    clearTimeout(timeout);
                    timeout = undefined;
                }
                console.warn("[" + accessoryName + "] Unhandled error thrown inside snapshot request handler: " + error.stack);
                reject(-70402 /* SERVICE_COMMUNICATION_FAILURE */);
            }
        });
    };
    /**
     * @private
     */
    CameraController.prototype.handleCloseConnection = function (sessionID) {
        if (this.delegate instanceof camera_1.LegacyCameraSourceAdapter) {
            this.delegate.forwardCloseConnection(sessionID);
        }
    };
    CameraController.STREAM_MANAGEMENT = "streamManagement"; // key to index all RTPStreamManagement services
    return CameraController;
}(events_1.EventEmitter));
exports.CameraController = CameraController;
//# sourceMappingURL=CameraController.js.map