/*********
  © Adrian Zilly
  elektro&it@zapadi.net

  Projekt // Backofen Temperatur Standalone Home.app 
  Hardware Lib: "Generic ESP8622 Module"

  v2.1 (Aktuellste) 20240301
  ---------Historie-------------
  v2.1 - WLAN Auto Reconnect eingefuegt
  v2.0 - Temperaturbereiche >100 und <0 Grad auf bis 300 und -100 geaendert.
  v1.8 - Finalisierung
  v1.7 - Sensor auslesen eingefuegt
  V1.6 - Grund Script erstellt
*********/

#include <Arduino.h>
#include <ESP8266WiFi.h>
#include "max6675.h"

#include <arduino_homekit_server.h>
#include <homekit/types.h>
#include <homekit/characteristics.h>

#define PL(s) Serial.println(s)
#define P(s) Serial.print(s)

//D0 16 //led
//D3  0 //flash button
//D4  2 //led

#define PIN_LED 16//D0

int thermoDO = 12; //D6 - Thermosensor S0
int thermoCS = 13; //D7 - CS
int thermoCLK = 14; //D5 - SCK

MAX6675 thermocouple(thermoCLK, thermoCS, thermoDO);

//User Anpassungen
float GrenzwertTemp = 150; //Benachrichtigen bei dieser Temperatur (Ausloesen Anwesenheitssensor) wird zurueckgesetzt bei 20 Grad weniger wie der eingestellte Wert.
const char *ssid = ""; //WLAN NAME
const char *password = ""; //WLAN PASSWORT

void setup() {
	Serial.begin(115200);
	Serial.setRxBufferSize(32);
	Serial.setDebugOutput(false);

	pinMode(PIN_LED, OUTPUT);
	WiFi.mode(WIFI_STA);
	WiFi.persistent(false);
	WiFi.disconnect(false);
	WiFi.setAutoReconnect(true);
	WiFi.begin(ssid, password);

	printf("\n");
	printf("SketchSize: %d B\n", ESP.getSketchSize());
	printf("FreeSketchSpace: %d B\n", ESP.getFreeSketchSpace());
	printf("FlashChipSize: %d B\n", ESP.getFlashChipSize());
	printf("FlashChipRealSize: %d B\n", ESP.getFlashChipRealSize());
	printf("FlashChipSpeed: %d\n", ESP.getFlashChipSpeed());
	printf("SdkVersion: %s\n", ESP.getSdkVersion());
	printf("FullVersion: %s\n", ESP.getFullVersion().c_str());
	printf("CpuFreq: %dMHz\n", ESP.getCpuFreqMHz());
	printf("FreeHeap: %d B\n", ESP.getFreeHeap());
	printf("ResetInfo: %s\n", ESP.getResetInfo().c_str());
	printf("ResetReason: %s\n", ESP.getResetReason().c_str());
	DEBUG_HEAP();
	homekit_setup();
	DEBUG_HEAP();
}

void WLANCheck(){
  int wifi_retry = 0;
  while(WiFi.status() != WL_CONNECTED && wifi_retry < 5 ) {
      wifi_retry++;
      INFO("WiFi not connected. Try to reconnect. Try-Counter: %d", wifi_retry);
      WiFi.disconnect();
      WiFi.mode(WIFI_OFF);
      delay(100);
      WiFi.mode(WIFI_STA);
      WiFi.persistent(false);
      WiFi.disconnect(false);
      WiFi.setAutoReconnect(true);
      WiFi.begin(ssid, password);
      delay(1000);
  }
  if(wifi_retry >= 5) {
      INFO("WiFi not connected. REBOOT!");
      ESP.restart();
  }
}

void loop() {
	homekit_loop();
	delay(5);
}

//==============================
// Homekit setup and loop
//==============================

extern "C" homekit_server_config_t config;
extern "C" homekit_characteristic_t name;
extern "C" void accessory_init();
extern "C" homekit_characteristic_t cha_current_temperature;
extern "C" homekit_characteristic_t occupancy_detected;

uint32_t next_heap_millis = 0;

void homekit_setup() {
	accessory_init();
	uint8_t mac[WL_MAC_ADDR_LENGTH];
	WiFi.macAddress(mac);
	int name_len = snprintf(NULL, 0, "%s_%02X%02X%02X", name.value.string_value, mac[3], mac[4], mac[5]);
	char *name_value = (char*)malloc(name_len + 1);
	snprintf(name_value, name_len + 1, "%s_%02X%02X%02X", name.value.string_value, mac[3], mac[4], mac[5]);
	name.value = HOMEKIT_STRING_CPP(name_value);
  *cha_current_temperature.min_value = -100;
  *cha_current_temperature.max_value = 300;

	arduino_homekit_setup(&config);
 
}

void homekit_loop() {
	arduino_homekit_loop();
	uint32_t time = millis();
	if (time > next_heap_millis) {
		INFO("heap: %d, sockets: %d", ESP.getFreeHeap(), arduino_homekit_connected_clients_count());
		next_heap_millis = time + 5000;
    GetTempThermo();
    WLANCheck();
	}
}

void GetTempThermo(){
  float temperature_value = thermocouple.readCelsius(); 
  INFO("Temperatur: %d", temperature_value);

  if (temperature_value > 300 or temperature_value < -99) {
    temperature_value = 0;
  }

  cha_current_temperature.value.float_value = temperature_value;
  homekit_characteristic_notify(&cha_current_temperature, cha_current_temperature.value);
  
  if (temperature_value >= GrenzwertTemp){
    occupancy_detected.value.uint8_value = float(1);
    homekit_characteristic_notify(&occupancy_detected, occupancy_detected.value);
  }
  else if (temperature_value < GrenzwertTemp - 20){
    occupancy_detected.value.uint8_value = float(0);
    homekit_characteristic_notify(&occupancy_detected, occupancy_detected.value);
  }
}
